"""
sync.py

Implementation of the Sync operator C for the tick‑flip algebra.

The sync operator projects a tick distribution onto the symmetric subspace,
defined by C|ψ⟩ = ½(|ψ⟩ + reversal(|ψ⟩)).  That is, it averages the
distribution with its reversed copy to enforce symmetry.
"""

import numpy as np
from .tick_state import TickState


def C(state: TickState) -> TickState:
    """
    Apply the Sync operator to a tick state.

    Parameters
    ----------
    state : TickState
        The current tick state, with distribution array and N.

    Returns
    -------
    TickState
        A new tick state whose distribution is the symmetric projection:
        new_dist = 0.5 * (dist + dist[::-1]).
    """
    dist = state.distribution
    rev = dist[::-1]
    new_dist = 0.5 * (dist + rev)
    # Return a new TickState; N is preserved
    return TickState(new_dist, state.N)
